// ****************************************************************************
// GlottalImageExplorer.
// Copyright (C) 2015-2016 Peter Birkholz.
// This program is free and open-source software.
// ****************************************************************************

#include "SignalPicture.h"

// ****************************************************************************
// IDs.
// ****************************************************************************

static const int IDM_ZOOM_IN = 9000;
static const int IDM_ZOOM_OUT = 9001;
static const int IDM_MAX_VALUE_4000 = 9002;
static const int IDM_MAX_VALUE_8000 = 9003;
static const int IDM_MAX_VALUE_16000 = 9004;
static const int IDM_AUTOSCALE = 9005;

// ****************************************************************************
// The event table.
// ****************************************************************************

BEGIN_EVENT_TABLE(SignalPicture, BasicPicture)
  EVT_MOUSE_EVENTS(OnMouseEvent)
  EVT_MENU(IDM_ZOOM_IN, OnZoomIn)
  EVT_MENU(IDM_ZOOM_OUT, OnZoomOut)
  EVT_MENU(IDM_MAX_VALUE_4000, OnMaxValue4000)
  EVT_MENU(IDM_MAX_VALUE_8000, OnMaxValue8000)
  EVT_MENU(IDM_MAX_VALUE_16000, OnMaxValue16000)
  EVT_MENU(IDM_AUTOSCALE, OnAutoscale)
END_EVENT_TABLE()


// ****************************************************************************
/// Construcor.
// ****************************************************************************

SignalPicture::SignalPicture(wxWindow *parent) : BasicPicture(parent)
{
  numVisFrames = 512;
  valueScale = MAX_VALUE_4000;

  // ****************************************************************
  // Init the graph.
  // ****************************************************************

  graph.init(this, 60, 5, 0, 25);
  graph.initAbscissa(PQ_INDEX, 0.0, 1.0,
    -1000.0, -10.0, -10.0,
    1000, 10, 10,
    5, 0, true, false, false);

  graph.initLinearOrdinate(PQ_COUNT, 0.0, 1.0,
    0.0, 0.0, 0.0,
    Film::WIDTH * Film::HEIGHT, Film::WIDTH * Film::HEIGHT, Film::WIDTH * Film::HEIGHT,
    4, 0, true, false, true);

  graph.isLinearOrdinate = true;

  // ****************************************************************
  // Create the context menu.
  // ****************************************************************

  contextMenu = new wxMenu();
  contextMenu->Append(IDM_ZOOM_IN, "Zoom in");
  contextMenu->Append(IDM_ZOOM_OUT, "Zoom out");
  contextMenu->AppendSeparator();
  contextMenu->AppendRadioItem(IDM_MAX_VALUE_4000, "Max. value = 4000");
  contextMenu->AppendRadioItem(IDM_MAX_VALUE_8000, "Max. value = 8000");
  contextMenu->AppendRadioItem(IDM_MAX_VALUE_16000, "Max. value = 16000");
  contextMenu->AppendRadioItem(IDM_AUTOSCALE, "Autoscale");

  menuX = 0;
  menuY = 0;
}


// ****************************************************************************
/// Draws the picture.
// ****************************************************************************

void SignalPicture::draw(wxDC &dc)
{
  int i;
  int x;
  int graphX, graphY, graphW, graphH;
  graph.getDimensions(graphX, graphY, graphW, graphH);

  Data *data = Data::getInstance();

  // Clear the background

  dc.SetBackground(*wxWHITE_BRUSH);
  dc.Clear();

  int numFrames = data->film->getNumFrames();

  if (numFrames < 1)
  {
    dc.DrawText("No film loaded.", 3, 3);
    return;
  }

  // ****************************************************************
  // Determine the abscissa limits.
  // ****************************************************************

  int currentFrame = data->getCurrFrameIndex(0);
  int firstFrame = currentFrame - numVisFrames / 2;

  if ((firstFrame + numVisFrames >= numFrames) && (numFrames > numVisFrames))
  {
    firstFrame = numFrames - numVisFrames;
  }

  if (firstFrame < 0)
  {
    firstFrame = 0;
  }

  int lastFrame = firstFrame + numVisFrames - 1;
  if (lastFrame >= numFrames)
  {
    lastFrame = numFrames - 1;
  }

  // ****************************************************************
  // Determine the maximum value of the visible frames.
  // ****************************************************************

  int maxValue = 1;   // Prevent division by zero
  for (i = firstFrame; i < lastFrame; i++)
  {
    if (data->srgData[i].glottisArea_pix > maxValue)
    {
      maxValue = data->srgData[i].glottisArea_pix;
    }
  }

  // ****************************************************************
  // Set the axes limits and draw the axes.
  // ****************************************************************

  graph.abscissa.negativeLimit = firstFrame;
  graph.abscissa.positiveLimit = firstFrame + numVisFrames - 1;
  graph.paintAbscissa(dc);

  graph.linearOrdinate.negativeLimit = 0;
  graph.linearOrdinate.positiveLimit = maxValue;  // Corresponds to "autoscale"

  if (valueScale == MAX_VALUE_4000)
  {
    graph.linearOrdinate.positiveLimit = 4000;
  }
  else
  if (valueScale == MAX_VALUE_8000)
  {
    graph.linearOrdinate.positiveLimit = 8000;
  }
  else
  if (valueScale == MAX_VALUE_16000)
  {
    graph.linearOrdinate.positiveLimit = 16000;
  }

  graph.paintOrdinate(dc);

  // ****************************************************************
  // Draw color lines at all frames where the user defined the 
  // segmentation.
  // ****************************************************************

  dc.SetPen(*wxRED_PEN);

  for (i = 0; i < numFrames; i++)
  {
    if (data->srgData[i].isUserDefinedSegmentation)
    {
      x = graph.getXPos(i);

      if ((x >= graphX) && (x < graphX + graphW))
      {
        dc.DrawLine(x, graphY, x, graphY + graphH);
      }
    }
  }

  // ****************************************************************
  // Draw a black line at the current frame.
  // ****************************************************************

  dc.SetPen(*wxBLACK_PEN);
  x = graph.getXPos(currentFrame);

  if ((x >= graphX) && (x < graphX + graphW))
  {
    dc.DrawLine(x, graphY, x, graphY + graphH);
  }
  
  // ****************************************************************
  // Draw the values of all visible frames and connect them with
  // lines.
  // ****************************************************************

  dc.SetPen(*wxBLACK_PEN);

  int x0, y0, x1, y1;
  
  for (i = firstFrame; i < lastFrame - 1; i++)
  {
    x0 = graph.getXPos(i);
    y0 = graph.getYPos(data->srgData[i].glottisArea_pix);

    x1 = graph.getXPos(i + 1);
    y1 = graph.getYPos(data->srgData[i + 1].glottisArea_pix);

    dc.DrawLine(x0, y0, x1, y1);
  }

}



// ****************************************************************************
// ****************************************************************************

void SignalPicture::OnMouseEvent(wxMouseEvent &event)
{
  int areaWidth, areaHeight;
  this->GetSize(&areaWidth, &areaHeight);

  int mx = event.GetX();
  int my = event.GetY();

  // ****************************************************************
  // The right mouse button changed to down.
  // ****************************************************************

  if (event.RightDown())
  {
    menuX = mx;
    menuY = my;

    if (valueScale == MAX_VALUE_4000) { contextMenu->Check(IDM_MAX_VALUE_4000, true); }
    else
    if (valueScale == MAX_VALUE_8000) { contextMenu->Check(IDM_MAX_VALUE_8000, true); }
    else
    if (valueScale == MAX_VALUE_16000) { contextMenu->Check(IDM_MAX_VALUE_16000, true); }
    else
    if (valueScale == AUTOSCALE) { contextMenu->Check(IDM_AUTOSCALE, true); }

    PopupMenu(contextMenu);
    return;
  }
}

// ****************************************************************************
// ****************************************************************************

void SignalPicture::OnZoomIn(wxCommandEvent &event)
{
  if (numVisFrames > 32)
  {
    numVisFrames /= 2;
  }
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SignalPicture::OnZoomOut(wxCommandEvent &event)
{
  if (numVisFrames < 2048)
  {
    numVisFrames *= 2;
  }
  this->Refresh();
}

// ****************************************************************************
// ****************************************************************************

void SignalPicture::OnMaxValue4000(wxCommandEvent &event)
{
  valueScale = MAX_VALUE_4000;
  this->Refresh();
}

// ****************************************************************************
// ****************************************************************************

void SignalPicture::OnMaxValue8000(wxCommandEvent &event)
{
  valueScale = MAX_VALUE_8000;
  this->Refresh();
}

// ****************************************************************************
// ****************************************************************************

void SignalPicture::OnMaxValue16000(wxCommandEvent &event)
{
  valueScale = MAX_VALUE_16000;
  this->Refresh();
}

// ****************************************************************************
// ****************************************************************************

void SignalPicture::OnAutoscale(wxCommandEvent &event)
{
  valueScale = AUTOSCALE;
  this->Refresh();
}


// ****************************************************************************
